/*
  ==============================================================================

    SonicCrypt Flux Gate
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"
#include "FluxEngine.h"

SonicCryptFluxGateAudioProcessor::SonicCryptFluxGateAudioProcessor()
#ifndef JucePlugin_PreferredChannelConfigurations
    : AudioProcessor(BusesProperties()
        .withInput("Input", juce::AudioChannelSet::stereo(), true)
        .withOutput("Output", juce::AudioChannelSet::stereo(), true)
    ),
    apvts(*this, nullptr, "Parameters", createParameterLayout())
#endif
{
    auto folder = getPresetFolder();
    if (!folder.exists()) folder.createDirectory();

    lastSteps = 16;
    lastPulses = 8;
    lastRot = 0;

    // Initialisiere Gedächtnis mit voller Lautstärke
    velocityMap.assign(32, 1.0f);
    updatePattern(lastSteps, lastPulses, lastRot);
}

SonicCryptFluxGateAudioProcessor::~SonicCryptFluxGateAudioProcessor() {}

juce::AudioProcessorValueTreeState::ParameterLayout SonicCryptFluxGateAudioProcessor::createParameterLayout()
{
    juce::AudioProcessorValueTreeState::ParameterLayout layout;

    layout.add(std::make_unique<juce::AudioParameterInt>("steps", "Steps", 2, 32, 16));
    layout.add(std::make_unique<juce::AudioParameterInt>("pulses", "Density", 0, 32, 8));
    layout.add(std::make_unique<juce::AudioParameterInt>("rotation", "Shift", -16, 16, 0));
    layout.add(std::make_unique<juce::AudioParameterFloat>("swing", "Swing", 0.0f, 1.0f, 0.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("glitch", "Chaos", 0.0f, 1.0f, 0.0f));
    layout.add(std::make_unique<juce::AudioParameterInt>("rate", "Rate", 0, 3, 2));
    layout.add(std::make_unique<juce::AudioParameterFloat>("smooth", "Smoothness", 0.0f, 200.0f, 10.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("mix", "Mix", 0.0f, 1.0f, 1.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("width", "Stereo Offset", 0.0f, 16.0f, 0.0f));

    return layout;
}

// --- LOGIC: PATTERN UPDATE WITH MEMORY ---
void SonicCryptFluxGateAudioProcessor::updatePattern(int steps, int pulses, int rot)
{
    // 1. Berechne Rhythmus (An/Aus)
    std::vector<float> rhythm = FluxEngine::generateEuclidean(steps, pulses, rot);

    // 2. Wende Gedächtnis an
    currentPattern.resize(steps);
    if (velocityMap.size() < 32) velocityMap.resize(32, 1.0f);

    for (int i = 0; i < steps; ++i) {
        if (rhythm[i] > 0.5f) {
            // Wenn Step aktiv, hole Lautstärke aus Memory
            float memVal = velocityMap[i];
            currentPattern[i] = (memVal > 0.01f) ? memVal : 1.0f;
        }
        else {
            currentPattern[i] = 0.0f;
        }
    }

    lastSteps = steps;
    lastPulses = pulses;
    lastRot = rot;
}

void SonicCryptFluxGateAudioProcessor::triggerFluxRandomization()
{
    juce::Random rng;

    int newSteps = rng.nextInt(30) + 2;
    int newPulses = rng.nextInt(newSteps) + 1;
    int newRot = rng.nextInt(32) - 16;

    float newSwing = rng.nextFloat() * 0.4f;
    float newChaos = (rng.nextFloat() > 0.6f) ? (rng.nextFloat() * 0.4f) : 0.0f;
    int newRate = rng.nextInt(4);

    float newSmooth = rng.nextFloat() * 100.0f;
    float newMix = 0.5f + (rng.nextFloat() * 0.5f);
    float newWidth = rng.nextFloat() * 8.0f;

    apvts.getParameter("steps")->setValueNotifyingHost(apvts.getParameter("steps")->convertTo0to1(newSteps));
    apvts.getParameter("pulses")->setValueNotifyingHost(apvts.getParameter("pulses")->convertTo0to1(newPulses));
    apvts.getParameter("rotation")->setValueNotifyingHost(apvts.getParameter("rotation")->convertTo0to1(newRot));
    apvts.getParameter("swing")->setValueNotifyingHost(newSwing);
    apvts.getParameter("glitch")->setValueNotifyingHost(newChaos);
    apvts.getParameter("rate")->setValueNotifyingHost(apvts.getParameter("rate")->convertTo0to1(newRate));
    apvts.getParameter("smooth")->setValueNotifyingHost(apvts.getParameter("smooth")->convertTo0to1(newSmooth));
    apvts.getParameter("mix")->setValueNotifyingHost(newMix);
    apvts.getParameter("width")->setValueNotifyingHost(apvts.getParameter("width")->convertTo0to1(newWidth));

    // FLUX: Fülle Gedächtnis mit neuen Zufallswerten
    for (int i = 0; i < 32; ++i) {
        velocityMap[i] = 0.6f + (rng.nextFloat() * 0.4f);
    }

    updatePattern(newSteps, newPulses, newRot);

    // Step 1 immer aktiv nach Flux (für sofortiges Feedback)
    if (!currentPattern.empty()) currentPattern[0] = velocityMap[0];
}

void SonicCryptFluxGateAudioProcessor::prepareToPlay(double sampleRate, int samplesPerBlock)
{
    float initialSmooth = *apvts.getRawParameterValue("smooth");
    float safeSmooth = juce::jmax(0.002f, initialSmooth / 1000.0f);
    currentSmoothMs = initialSmooth;

    gateSmootherL.reset(sampleRate, safeSmooth);
    gateSmootherL.setCurrentAndTargetValue(1.0f);
    gateSmootherR.reset(sampleRate, safeSmooth);
    gateSmootherR.setCurrentAndTargetValue(1.0f);

    beatPhasor = 0.0;
    isTriggered = false;
    releaseTimer = 0;
    activeMidiNotes = 0;

    isSilent = true;
    silenceCounter = 0;

    stutterBuffer.setSize(2, (int)sampleRate);
    stutterBuffer.clear();
    stutterWritePos = 0;

    isStuttering = false;
    stutterReadOffset = 0;
    stutterCooldown = 0;
    stutterFade = 0.0f;

    if (currentPattern.empty()) {
        updatePattern(16, 8, 0);
    }
}

void SonicCryptFluxGateAudioProcessor::releaseResources() {}

void SonicCryptFluxGateAudioProcessor::togglePatternStep(int stepIndex)
{
    if (stepIndex >= 0 && stepIndex < (int)currentPattern.size()) {
        if (currentPattern[stepIndex] > 0.01f) {
            currentPattern[stepIndex] = 0.0f;
        }
        else {
            // Restore from Memory
            float val = velocityMap[stepIndex];
            currentPattern[stepIndex] = (val > 0.01f) ? val : 1.0f;
        }
    }
}

void SonicCryptFluxGateAudioProcessor::setPatternStepLevel(int stepIndex, float level)
{
    if (stepIndex >= 0 && stepIndex < (int)currentPattern.size()) {
        float clamped = juce::jlimit(0.0f, 1.0f, level);
        currentPattern[stepIndex] = clamped;

        // WICHTIG: Speichere Änderung im Gedächtnis
        if (stepIndex < 32) velocityMap[stepIndex] = clamped;
    }
}

void SonicCryptFluxGateAudioProcessor::processBlock(juce::AudioBuffer<float>& buffer, juce::MidiBuffer& midiMessages)
{
    juce::ScopedNoDenormals noDenormals;
    auto totalNumInputChannels = getTotalNumInputChannels();
    auto totalNumOutputChannels = getTotalNumOutputChannels();

    for (auto i = totalNumInputChannels; i < totalNumOutputChannels; ++i)
        buffer.clear(i, 0, buffer.getNumSamples());

    // --- MIDI ANALYSE (RESET DETECTOR) ---
    bool noteTriggered = false;
    for (const auto metadata : midiMessages) {
        auto msg = metadata.getMessage();
        if (msg.isNoteOn()) {
            activeMidiNotes++;
            noteTriggered = true; // NEUE NOTE -> RESET BEFEHL
        }
        else if (msg.isNoteOff()) {
            activeMidiNotes--;
            if (activeMidiNotes < 0) activeMidiNotes = 0;
        }
        else if (msg.isAllNotesOff()) {
            activeMidiNotes = 0;
        }
    }

    // --- AUDIO ANALYSE ---
    float inputMag = buffer.getMagnitude(0, buffer.getNumSamples());
    bool audioActive = inputMag > 0.002f;

    // --- TRIGGER LOGIC ---
    bool shouldRun = (activeMidiNotes > 0) || audioActive;

    if (shouldRun) {
        // RESET Bedingungen:
        // 1. Wenn wir aus der Stille kommen (!isTriggered)
        // 2. Wenn eine neue MIDI Note gedrückt wurde (noteTriggered)
        if (!isTriggered || noteTriggered) {
            beatPhasor = 0.0;       // HARD RESET auf 0
            stutterWritePos = 0;
            isTriggered = true;

            // Soft-Reset der Smoother für knackigen Attack
            if (!currentPattern.empty()) {
                float startVal = currentPattern[0];
                gateSmootherL.setCurrentAndTargetValue(startVal);
                gateSmootherR.setCurrentAndTargetValue(startVal);
            }
        }
        releaseTimer = 0;
    }
    else {
        // Kein Signal -> Release Timer
        if (isTriggered) {
            releaseTimer += buffer.getNumSamples();
            if (releaseTimer > (getSampleRate() * 0.5)) {
                isTriggered = false;
            }
        }
    }

    // --- PARAMETER ---
    int rawSteps = (int)*apvts.getRawParameterValue("steps");
    int rawPulses = (int)*apvts.getRawParameterValue("pulses");
    int rot = (int)*apvts.getRawParameterValue("rotation");
    float swing = *apvts.getRawParameterValue("swing");
    float chaos = *apvts.getRawParameterValue("glitch");
    int rateVal = (int)*apvts.getRawParameterValue("rate");
    float smoothMs = *apvts.getRawParameterValue("smooth");
    float mix = *apvts.getRawParameterValue("mix");
    float width = *apvts.getRawParameterValue("width");

    // Pattern Logic (WICHTIG: updatePattern benutzen!)
    int activeSteps = rawSteps;
    int activePulses = rawPulses;
    if (activePulses > activeSteps) activePulses = activeSteps;

    if (activeSteps != lastSteps || activePulses != lastPulses || rot != lastRot) {
        // Hier rufen wir jetzt die Memory-Funktion auf
        updatePattern(activeSteps, activePulses, rot);
    }

    // BPM Sync
    double bpm = 120.0;
    if (auto* ph = getPlayHead()) {
        if (auto pos = ph->getPosition()) {
            if (pos->getBpm().hasValue()) bpm = *pos->getBpm();
        }
    }
    double rateFactor = 1.0;
    if (rateVal == 1) rateFactor = 2.0;
    if (rateVal == 2) rateFactor = 4.0;
    if (rateVal == 3) rateFactor = 8.0;
    double beatsPerSample = (bpm / 60.0) / getSampleRate();

    // Smoother Logic
    if (std::abs(currentSmoothMs - smoothMs) > 0.1f) {
        float currentL = gateSmootherL.getCurrentValue();
        float currentR = gateSmootherR.getCurrentValue();
        float safeSmooth = juce::jmax(0.002f, smoothMs / 1000.0f);
        gateSmootherL.reset(getSampleRate(), safeSmooth);
        gateSmootherR.reset(getSampleRate(), safeSmooth);
        gateSmootherL.setCurrentAndTargetValue(currentL);
        gateSmootherR.setCurrentAndTargetValue(currentR);
        currentSmoothMs = smoothMs;
    }

    // --- AUDIO PROCESSING LOOP ---
    juce::Random& rng = juce::Random::getSystemRandom();

    for (int s = 0; s < buffer.getNumSamples(); ++s) {

        float inL = buffer.getSample(0, s);
        float inR = (totalNumInputChannels > 1) ? buffer.getSample(1, s) : inL;

        // Buffer Write
        stutterBuffer.setSample(0, stutterWritePos, inL);
        if (stutterBuffer.getNumChannels() > 1) stutterBuffer.setSample(1, stutterWritePos, inR);
        stutterWritePos++;
        if (stutterWritePos >= stutterBuffer.getNumSamples()) stutterWritePos = 0;

        // --- PHASOR UPDATE & SWING ---
        double effectivePos = 0.0;

        if (isTriggered) {
            double rawGridPos = beatPhasor * rateFactor;
            beatPhasor += beatsPerSample;

            double integral;
            std::modf(rawGridPos, &integral);
            long stepCount = (long)integral;

            double swungPos = rawGridPos;

            // SWING (Boosted)
            if (swing > 0.01f && stepCount % 2 != 0) {
                swungPos -= (swing * 0.45f);
            }

            effectivePos = swungPos;
        }

        // Stutter
        float currentSignalL = inL;
        float currentSignalR = inR;

        if (chaos > 0.0f && isTriggered) {
            if (isStuttering) {
                int readPos = (stutterWritePos - stutterLoopLength) + stutterReadOffset;
                while (readPos < 0) readPos += stutterBuffer.getNumSamples();
                while (readPos >= stutterBuffer.getNumSamples()) readPos -= stutterBuffer.getNumSamples();

                float loopL = stutterBuffer.getSample(0, readPos);
                float loopR = (stutterBuffer.getNumChannels() > 1) ? stutterBuffer.getSample(1, readPos) : loopL;

                int fadeLen = 200;
                float windowGain = 1.0f;
                if (stutterReadOffset < fadeLen)
                    windowGain = (float)stutterReadOffset / (float)fadeLen;
                else if (stutterReadOffset > (stutterLoopLength - fadeLen))
                    windowGain = (float)(stutterLoopLength - stutterReadOffset) / (float)fadeLen;

                currentSignalL = loopL * windowGain;
                currentSignalR = loopR * windowGain;

                stutterReadOffset++;
                if (stutterReadOffset >= stutterLoopLength) stutterReadOffset = 0;

                stutterCooldown--;
                if (stutterCooldown <= 0) {
                    isStuttering = false;
                    stutterCooldown = (int)(getSampleRate() * 0.5f);
                }
            }
            else {
                stutterCooldown--;
                if (stutterCooldown < 0) {
                    if (rng.nextFloat() < (chaos * 0.0005f)) {
                        isStuttering = true;
                        int ms = 30 + rng.nextInt(70);
                        stutterLoopLength = (int)(getSampleRate() * (ms / 1000.0f));
                        stutterReadOffset = 0;
                        stutterCooldown = stutterLoopLength * (2 + rng.nextInt(6));
                    }
                }
            }
        }
        else {
            isStuttering = false;
        }

        // TARGET CALCULATION WITH GAP
        float targetL = 1.0f;
        float targetR = 1.0f;

        if (isTriggered) {
            int stepL = (int)std::floor(effectivePos) % activeSteps;
            if (stepL < 0) stepL += activeSteps;
            currentStepIndex = stepL;

            // GAP Logic (damit Swing besser hörbar ist)
            double posInStep = effectivePos - std::floor(effectivePos);
            float gateOpen = (posInStep < 0.90) ? 1.0f : 0.0f;

            if (stepL < (int)currentPattern.size())
                targetL = currentPattern[stepL] * gateOpen;

            if (width > 0.1f) {
                int stepR = (stepL + (int)width) % activeSteps;
                if (stepR < (int)currentPattern.size())
                    targetR = currentPattern[stepR] * gateOpen;
            }
            else { targetR = targetL; }
        }
        else {
            currentStepIndex = -1;
            targetL = 0.0f; targetR = 0.0f;
        }

        gateSmootherL.setTargetValue(targetL);
        gateSmootherR.setTargetValue(targetR);
        float envL = gateSmootherL.getNextValue();
        float envR = gateSmootherR.getNextValue();

        float processedL = currentSignalL * envL;
        float processedR = currentSignalR * envR;

        buffer.setSample(0, s, inL * (1.0f - mix) + processedL * mix);
        if (buffer.getNumChannels() > 1)
            buffer.setSample(1, s, inR * (1.0f - mix) + processedR * mix);
    }
}

// ... STANDARD FUNCTIONS ...
juce::File SonicCryptFluxGateAudioProcessor::getPresetFolder() const {
    auto docDir = juce::File::getSpecialLocation(juce::File::userDocumentsDirectory);
    return docDir.getChildFile("SonicCrypt").getChildFile("FluxGate").getChildFile("Presets");
}
void SonicCryptFluxGateAudioProcessor::savePreset(const juce::String& name) {
    auto folder = getPresetFolder(); if (!folder.exists()) folder.createDirectory();
    auto file = folder.getChildFile(name).withFileExtension("scp");
    juce::FileOutputStream out(file);
    if (out.openedOk()) {
        out.writeString("SCFLUX1");
        juce::MemoryBlock paramBlock; { juce::MemoryOutputStream mos(paramBlock, false); apvts.state.writeToStream(mos); }
        out.writeInt64(paramBlock.getSize()); out.write(paramBlock.getData(), paramBlock.getSize());
    }
}
void SonicCryptFluxGateAudioProcessor::loadPresetFile(const juce::File& file) {
    if (!file.existsAsFile()) return;
    juce::FileInputStream in(file);
    if (in.openedOk()) {
        if (in.readString() != "SCFLUX1") return;
        juce::int64 paramSize = in.readInt64();
        if (paramSize > 0 && paramSize < 1024 * 1024) {
            juce::MemoryBlock paramBlock; in.readIntoMemoryBlock(paramBlock, (size_t)paramSize);
            juce::MemoryInputStream mis(paramBlock, false);
            auto state = juce::ValueTree::readFromStream(mis); if (state.isValid()) apvts.replaceState(state);
        }
    }
}
void SonicCryptFluxGateAudioProcessor::getStateInformation(juce::MemoryBlock& destData) {
    juce::MemoryOutputStream out(destData, false); out.writeString("SCFLUXState");
    auto state = apvts.copyState();
    state.setProperty("uiWidth", lastUIWidth, nullptr); state.setProperty("uiHeight", lastUIHeight, nullptr);
    state.writeToStream(out);
}
void SonicCryptFluxGateAudioProcessor::setStateInformation(const void* data, int sizeInBytes) {
    juce::MemoryInputStream in(data, sizeInBytes, false);
    if (in.readString() == "SCFLUXState") {
        auto newState = juce::ValueTree::readFromStream(in);
        if (newState.isValid()) {
            if (newState.hasProperty("uiWidth")) lastUIWidth = newState.getProperty("uiWidth");
            if (newState.hasProperty("uiHeight")) lastUIHeight = newState.getProperty("uiHeight");
            apvts.replaceState(newState);
        }
    }
}
void SonicCryptFluxGateAudioProcessor::valueTreePropertyChanged(juce::ValueTree&, const juce::Identifier&) {}
bool SonicCryptFluxGateAudioProcessor::hasEditor() const { return true; }
juce::AudioProcessorEditor* SonicCryptFluxGateAudioProcessor::createEditor() { return new SonicCryptFluxGateAudioProcessorEditor(*this); }
const juce::String SonicCryptFluxGateAudioProcessor::getName() const { return "SonicCrypt Flux Gate"; }
bool SonicCryptFluxGateAudioProcessor::acceptsMidi() const { return true; }
bool SonicCryptFluxGateAudioProcessor::producesMidi() const { return false; }
bool SonicCryptFluxGateAudioProcessor::isMidiEffect() const { return false; }
double SonicCryptFluxGateAudioProcessor::getTailLengthSeconds() const { return 0.0; }
int SonicCryptFluxGateAudioProcessor::getNumPrograms() { return 1; }
int SonicCryptFluxGateAudioProcessor::getCurrentProgram() { return 0; }
void SonicCryptFluxGateAudioProcessor::setCurrentProgram(int) {}
const juce::String SonicCryptFluxGateAudioProcessor::getProgramName(int) { return {}; }
void SonicCryptFluxGateAudioProcessor::changeProgramName(int, const juce::String&) {}
bool SonicCryptFluxGateAudioProcessor::isBusesLayoutSupported(const BusesLayout& layouts) const {
    if (layouts.getMainOutputChannelSet() != juce::AudioChannelSet::mono() && layouts.getMainOutputChannelSet() != juce::AudioChannelSet::stereo()) return false;
    if (layouts.getMainOutputChannelSet() != layouts.getMainInputChannelSet()) return false;
    return true;
}
juce::AudioProcessor* JUCE_CALLTYPE createPluginFilter() { return new SonicCryptFluxGateAudioProcessor(); }